function body = importBody(dataFile)
arguments
    dataFile string = [];
end

if isempty(dataFile)
    [dataFileName, path] = uigetfile({'*.xlsx;*.xls',...
        'Excel Spreadsheet (*.xlsx,*.xls)';
        '*.*',  'All Files (*.*)'}, ...
        'Select a File');
    dataFile = fullfile(path, dataFileName);
end
[~, dataFileName] = fileparts(dataFile);

% Read main parameters from the "main" sheet
mainData = readcell(dataFile, 'Sheet', 'main');
mainData = cell2table(mainData(2:end, 2:end), 'VariableNames', mainData(1, 2:end), 'RowNames', mainData(2:end, 1));

body.shortName = dataFileName;
body.fullName = mainData.value{'Full name'}; % kg
body.class = mainData.value{'Class'};

rowNames = string(mainData.Properties.RowNames);

% Masses 
% Curb mass:
%   The user defines either curbMass or chassisMass
%       - If curbMass is specified, that is exactly the curb mass of the
%       vehicle and the components' masses are ignored. 
%       - If chassisMass is specified instead, the curb mass is the sum of
%       chassisMass and the components' masses.
% Load mass:
%   The user defines either maxLoad or grossMass
%       - If maxLoad is specified, that is the maximum payload. 
%       - If grossMass is specified, the maximum load mass is obtained
%       as the difference between (regulatory) gross weight and curb mass;
%       this is useful for HD applications
if ismember("Curb mass", rowNames)
    body.curbMass = mainData.value{"Curb mass"}; % kg
end
if ismember("Chassis mass", rowNames)
    body.chassisMass = mainData.value{"Chassis mass"}; % kg
end
if ismember("Max load mass", rowNames)
    body.maxLoad = mainData.value{"Max load mass"}; % kg
end
if ismember("Gross mass", rowNames)
    body.grossMass = mainData.value{"Gross mass"}; % kg
end

% Load axles data
if ismember("Axle specification", rowNames)
    body.axleMethod = mainData.value{'Axle specification'}; % -
else
    warning("Missing field ""Axle specification""; Value ""concentrated"" will be used.")
    body.axleMethod = "concentrated";
end

switch body.axleMethod
    case "explicit"
        axleData = readtable(dataFile, 'Sheet', 'axles', 'TextType', 'string');

        body.axles.relLoad = axleData.relLoad;
        body.axles.RRC_ISO = axleData.RRC_ISO;
        body.axles.Fz_ISO = axleData.Fz_ISO;
        body.axles.twinTyres = axleData.twinTyres;
        body.axles.tyre = axleData.tyre;
        body.axles.inertia = axleData.inertia;

        [C0] = rrcFromAxles(body.axles);
        body.rollResCoeff0 = C0;
        body.rollResCoeff1 = 0;

        body.numAxles = height(axleData);


    case "concentrated"
        body.numAxles = 2;
        body.axles.relLoad = [0.5 0.5];
        if ~ismember("Tyre", rowNames)
            error("Missing field ""Tyre"" to specify the tyre name.")
        end
        tyreName = string(mainData.value{"Tyre"});
        axleInertia = mainData.value{"Axle inertia"};
        body.axles.tyre = [tyreName, tyreName];
        body.axles.inertia = [axleInertia, axleInertia];

    otherwise
        error("The axle specification method must be ""explicit"" or ""concentrated""");

end

% Resistant forces
if ismember("Resistant forces method", rowNames)
    body.resForceMethod = mainData.value{'Resistant forces method'}; % -
else
    warning("Missing field ""Resistant forces method""; Value ""roadload"" will be used.")
    body.resForceMethod = "roadload";
end

switch body.resForceMethod
    case "roadload"
        body.f0 = mainData.value{'F0 (coast-down coefficient)'}; % N
        body.f1 = mainData.value{'F1 (coast-down coefficient)'}; % N / (m/s)
        body.f2 = mainData.value{'F2 (coast-down coefficient)'}; % N / (m/s)^2

    case "analytic"
        % F_res = F_roll + F_aero

        % Aerodynamic resistance
        % F_aero = 1/2 * airDensity * airDragCoef * frontArea * vehSpd^2
        if ismember("CdA", rowNames)
            body.CdA = mainData.value{'CdA'}; % m^2
            body.airDragCoef = 0;
            body.frontArea = 0;
        else
            body.airDragCoef = mainData.value{'Air drag coefficient'}; % -
            body.frontArea = mainData.value{'Front area'}; % m^2
            body.CdA = body.airDragCoef * body.frontArea;
        end

        % Rolling resistance
        switch body.axleMethod
            case "explicit"
                axleLoad = body.axles.relLoad;
                RRC = body.axles.RRC_ISO;
                Fz = body.axles.Fz_ISO;
                body.rollResCoeff0 = @(mass) sum( axleLoad .* RRC .* ( ( axleLoad .* mass .* 9.81 ) ./ ( 2 .* Fz ) ) .^ (-0.1) );
                body.rollResCoeff1 = 0;

            case "concentrated"
                % F_roll = rollResCoeff0 + rollResCoeff1 * vehSpd
                if ~ismember("Rolling resistance coeff. RRC0", rowNames) 
                    error("Missing field ""Rolling resistance coeff. RRC0"" (N).")
                end
                if ~ismember("Rolling resistance coeff. RRC1", rowNames) 
                    error("Missing field ""Rolling resistance coeff. RRC1"" N/(m/s).")
                end

                body.rollResCoeff0 = mainData.value{'Rolling resistance coeff. RRC0'}; % N
                body.rollResCoeff1 = mainData.value{'Rolling resistance coeff. RRC1'}; % N/(m/s)

        end

    otherwise
        error("The ""Resistant forces method"" must be ""roadload"" or ""analytic""");

end

end